function varargout = readTHFile(varargin)

%readTHFile   Reads a time-history data file
%
%  For VELOCITY DATA time-history files:
%    [u, v, w, ps, settings] = readTHFile(fileName, numSamplesToRead, offset)
%
%    Returns U, V, W, Pstatic and settings
%
%
%  For ANALOGUE DATA time-history files (such as pressure and voltage, etc.):
%    [data, settings] = readTHFile(fileName, numSamplesToRead, offset)
%
%    Returns data (one column per channel) and settings
%
%
%  Note: File name is optional - enter '' to get a file open dialog box
%	      Number of samples to read is optional - enter '0' to read the entire file
%        Offset is optional - enter number of samples to offset by (i.e. position of first sample to read)
%
%See also VELPITCHYAW, TRANSFORMAXES and ALIGNWITHFLOW


	numSamplesToRead = 0;
	offset = 0;

	% Get input arguments
	if nargin >= 1
		fileName = varargin{1};
	end
	if nargin >= 2
		numSamplesToRead = varargin{2};
	end
	if nargin >= 3
		offset = varargin{3};
	end

	% Show file open dialog if required
	if ~exist('fileName', 'var') || isempty(fileName)
		[fileName, filePath] = uigetfile({'*.th?;*.th;*.bp?', 'Supported TH files'}, 'Select time-history file to open...');
		if fileName == 0
			error('No file selected')
		end
		if filePath == 0
			error('No path selected')
		end
		fullFileName = strcat(filePath, fileName);
	end

	% Open file
	fid = fopen(fileName, 'rb');
	if fid < 3
		error('Failed to open time-history file')
	end

	% Read header
	fileFormat = fread(fid, 1, 'int32');
%	fprintf('File format : %i\n', fileFormat)

	switch fileFormat
		case {1, 6401}
			deviceID = fread(fid, 1, 'int32');
			deviceName = sprintf('Probe %0.3i', deviceID);
			deviceType = 4;  % Four-hole Cobra Probe
			dataType = 100;  % Velocity
			dateString = '';
			timeString = '';
			numSamples = fread(fid, 1, 'int32');
			blockSize = fread(fid, 1, 'int32');
			dataRate = fread(fid, 1, 'float64');
			hasPref = fread(fid, 1, 'uint8');

			if numSamples == 0
				numSamples = getNumSamplesPerChannel(fid, numChannels);
			end

%			fprintf('Device ID : %0.3i\n', deviceID)
%			fprintf('Num samples : %i\n', numSamples)
%			fprintf('Data rate : %0.1f Hz\n', dataRate)
%			if hasPref
%				fprintf('Has Pref : yes\n')
%			else
%				fprintf('Has Pref : no\n')
%			end
		case 2
			deviceID = fread(fid, 1, 'int32');
			deviceName = sprintf('Probe %0.3i', deviceID);
			deviceType = 4;  % Four-hole Cobra Probe
			dataType = 100;  % Velocity
			dateString = '';
			timeString = '';
			numSamples = fread(fid, 1, 'int32');
			blockSize = fread(fid, 1, 'int32');
			dataRate = fread(fid, 1, 'float64');
			pBaro = fread(fid, 1, 'float64');
			meanTemperature = fread(fid, 1, 'float64');
			hasPref = fread(fid, 1, 'uint8');

			if numSamples == 0
				numSamples = getNumSamplesPerChannel(fid, numChannels);
			end

%			fprintf('Probe ID : %0.3i\n', probeID)
%			fprintf('Num samples : %i\n', numSamples)
%			fprintf('Data rate : %0.1f Hz\n', dataRate)
%			fprintf('Barometric pressure : %0.1f Pa\n', pBaro)
%			fprintf('Mean temperature : %0.1f �C\n', meanTemperature)
%			if hasPref
%				fprintf('Has ref pressure : yes\n')
%			else
%				fprintf('Has ref pressure : no\n')
%			end
		case 3
			deviceType = fread(fid, 1, 'int32');
			deviceID = fread(fid, 1, 'int32');
			dataType = fread(fid, 1, 'int32');
			deviceName = sprintf('%s %0.3i', getDeviceTypeShortName(deviceType), deviceID);
			dateString = readDate(fid);
			timeString = readTime(fid);
			numSamples = fread(fid, 1, 'int32');
			blockSize = fread(fid, 1, 'int32');
			dataRate = fread(fid, 1, 'float64');
			pBaro = fread(fid, 1, 'float64');
			meanTemperature = fread(fid, 1, 'float64');
			hasPref = fread(fid, 1, 'uint8');

			if numSamples == 0
				numSamples = getNumSamplesPerChannel(fid, numChannels);
			end

%			fprintf('Device name         : %s\n', deviceName)
%			fprintf('Num samples         : %i\n', numSamples)
%			fprintf('Data rate           : %0.1f Hz\n', dataRate)
%			fprintf('Barometric pressure : %0.1f Pa\n', pBaro)
%			fprintf('Mean temperature    : %0.1f �C\n', meanTemperature)
%			if hasPref
%				fprintf('Has ref pressure    : yes\n')
%			else
%				fprintf('Has ref pressure    : no\n')
%			end
		case 101
			dataType = fread(fid, 1, 'int32');
			dataOrder = fread(fid, 1, 'int32');
			numChannels = fread(fid, 1, 'int32');
			numSamples = fread(fid, 1, 'int32');
			blockSize = fread(fid, 1, 'int32');
			dataRate = fread(fid, 1, 'float64');
			dateString = readDate(fid);
			timeString = readTime(fid);
			for n = 1:numChannels
				chanLabels(n) = {sprintf('Ch %d', n)};
				chanUnitAbrs(n) = {''};
			end

			if numSamples == 0
				numSamples = getNumSamplesPerChannel(fid, numChannels);
			end

%			fprintf('Num channels : %i\n', numChannels)
%			fprintf('Num samples  : %i\n', numSamples)
%			fprintf('Data rate    : %0.1f Hz\n', dataRate)
		case 102
			headerSize = fread(fid,1,'int32');
			dataType = fread(fid, 1, 'int32');
			dataOrder = fread(fid, 1, 'int32');
			numChannels = fread(fid, 1, 'int32');
			numSamples = fread(fid, 1, 'int32');
			blockSize = fread(fid, 1, 'int32');
			dataRate = fread(fid, 1, 'float64');
			dateString = readDate(fid);
			timeString = readTime(fid);
			for n = 1:numChannels
			   labelLength = fread(fid, 1, 'int32');
			   chanLabels(n) = { char( fread(fid, labelLength, 'uchar')' ) };
			end
			for n = 1:numChannels
			   unitAbrLength = fread(fid, 1, 'int32');
			   chanUnitAbrs(n) = { char( fread(fid, unitAbrLength, 'uchar')' ) };
			end

			if numSamples == 0
			   numSamples = getNumSamplesPerChannel(fid, numChannels);
			end

%			fprintf('Num channels : %i\n', numChannels)
%			fprintf('Num samples  : %i\n', numSamples)
%			fprintf('Data rate    : %0.1f Hz\n', dataRate)
		case 201
			deviceType = fread(fid, 1, 'int32');
			deviceID = fread(fid, 1, 'int32');
			deviceName = sprintf('%s %0.3i', getDeviceTypeShortName(deviceType), deviceID);
			dataType = fread(fid, 1, 'int32');
			dataOrder = fread(fid, 1, 'int32');
			numChannels = fread(fid, 1, 'int32');
			numSamples = fread(fid, 1, 'int32');
			blockSize = fread(fid, 1, 'int32');
			dataRate = fread(fid, 1, 'float64');
			dateString = readDate(fid);
			timeString = readTime(fid);
			for n = 1:numChannels
				chanLabels(n) = {sprintf('Ch %d', n)};
				chanUnitAbrs(n) = {''};
			end

			if numSamples == 0
			   numSamples = getNumSamplesPerChannel(fid, numChannels);
			end

%			fprintf('Device name  : %s\n', deviceName)
%			fprintf('Num channels : %i\n', numChannels)
%			fprintf('Num samples  : %i\n', numSamples)
%			fprintf('Data rate    : %0.1f Hz\n', dataRate)
		case 202
			headerSize = fread(fid, 1, 'int32');
			deviceNameLength = fread(fid, 1, 'int32');
			deviceName = char( fread(fid,deviceNameLength, 'uchar')' );
			deviceType = fread(fid, 1, 'int32');
			deviceID = fread(fid, 1, 'int32');
			dataType = fread(fid, 1, 'int32');
			dataOrder = fread(fid, 1, 'int32');
			numChannels = fread(fid, 1, 'int32');
			numSamples = fread(fid, 1, 'int32');
			blockSize = fread(fid, 1, 'int32');
			dataRate = fread(fid, 1, 'float64');
			dateString = readDate(fid);
			timeString = readTime(fid);
			for n = 1:numChannels
			   labelLength = fread(fid, 1, 'int32');
			   chanLabels(n) = { char( fread(fid, labelLength, 'uchar')' ) };
			end
			for n = 1:numChannels
			   unitAbrLength = fread(fid,1,'int32');
			   chanUnitAbrs(n) = { char( fread(fid, unitAbrLength, 'uchar')' ) };
			end

			if numSamples == 0
			   numSamples = getNumSamplesPerChannel(fid, numChannels);
			end

%			fprintf('Device name  : %s\n', deviceName)
%			fprintf('Num channels : %i\n', numChannels)
%			fprintf('Num samples  : %i\n', numSamples)
%			fprintf('Data rate    : %0.1f Hz\n', dataRate)
		otherwise
			error('File format not supported')
	end

%	fprintf('\n');

	% Configure settings output
	switch fileFormat
		case {1, 6401}
			varargout{5} = struct('deviceName',deviceName, 'deviceType',getDeviceTypeName(deviceType), 'deviceID',deviceID, ...
			                      'dataType',getDataTypeName(dataType), ...
			                      'blockSize',blockSize, 'dataRate',dataRate, ...
			                      'pRef',[], 'firstSampleDate',dateString, 'firstSampleTime',timeString);
		case {2, 3}
			varargout{5} = struct('deviceName',deviceName, 'deviceType',getDeviceTypeName(deviceType), 'deviceID',deviceID, ...
			                      'dataType',getDataTypeName(dataType), ...
			                      'blockSize',blockSize, 'dataRate',dataRate, ...
			                      'pBaro',pBaro, 'tMean',meanTemperature, ...
			                      'pRef',[], 'firstSampleDate',dateString, 'firstSampleTime',timeString);
		case {101, 102}
			varargout{2} = struct('dataType',getDataTypeName(dataType), 'dataOrder',getDataOrderName(dataOrder), ...
			                      'numChannels',numChannels, 'numSamples',numSamples, 'blockSize',blockSize, 'dataRate',dataRate, ...
			                      'firstSampleDate',dateString, 'firstSampleTime',timeString, ...
			                      'chanLabels',{chanLabels}', 'chanUnitAbrs',{chanUnitAbrs});
		case {201, 202}
			varargout{2} = struct('deviceName',deviceName, 'deviceType',getDeviceTypeName(deviceType), 'deviceID',deviceID, ...
			                      'dataType',getDataTypeName(dataType), 'dataOrder',getDataOrderName(dataOrder), ...
			                      'numChannels',numChannels, 'numSamples',numSamples, 'blockSize',blockSize, 'dataRate',dataRate, ...
			                      'firstSampleDate',dateString, 'firstSampleTime',timeString, ...
			                      'chanLabels',{chanLabels}', 'chanUnitAbrs',{chanUnitAbrs});
	end

	% Read data
	switch fileFormat
		case {1, 6401, 2, 3}
			% Seek to required start position
			if offset > numSamples
				error('Offset larger than number of samples in file')
			end
			blockStartSample = floor(offset/blockSize) * blockSize;
			if hasPref
				fseek(fid, blockStartSample*4*5, 'cof');
			else
				fseek(fid, blockStartSample*4*4, 'cof');
			end

			if numSamplesToRead == 0
				numBlocks = numSamples / blockSize;
			else
				numBlocks = floor(numSamplesToRead/blockSize);
			end

			varargout{1} = zeros(numBlocks*blockSize, 1);
			varargout{2} = zeros(numBlocks*blockSize, 1);
			varargout{3} = zeros(numBlocks*blockSize, 1);
			varargout{4} = zeros(numBlocks*blockSize, 1);
			if hasPref
				varargout{5}.pRef = zeros(numBlocks*blockSize, 1);
			end

			for blockNum = 1:numBlocks
				index1 = (blockNum-1) * blockSize + 1;
				index2 = blockNum * blockSize;

				varargout{1}(index1:index2) = fread(fid, blockSize, 'float32');
				varargout{2}(index1:index2) = fread(fid, blockSize, 'float32');
				varargout{3}(index1:index2) = fread(fid, blockSize, 'float32');
				varargout{4}(index1:index2) = fread(fid, blockSize, 'float32');
				if hasPref
					varargout{5}.pRef(index1:index2) = fread(fid, blockSize, 'float32');
				end
			end
		case {101, 102, 201, 202}
			% Seek to required start position
			if offset > numSamples
				error('Offset larger than number of samples in file')
			end
			blockStartSample = floor(offset/blockSize) * blockSize;
			fseek(fid, blockStartSample*numChannels*4, 'cof');

			if numSamplesToRead == 0
				numBlocks = numSamples / blockSize;
			else
				numBlocks = floor(numSamplesToRead/blockSize);
			end

			if dataOrder == 0   % doNonInterleaved
				varargout{1} = zeros(numBlocks*blockSize, numChannels);
				for blockNum = 1:numBlocks
					index1 = (blockNum-1) * blockSize + 1;
					index2 = blockNum * blockSize;

					varargout{1}(index1:index2,:) = fread(fid, [blockSize,numChannels], 'float32');
				end
			elseif dataOrder == 1   % doInterleaved
				varargout{1} = fread(fid, [numChannels,blockSize], 'float32')';
			end
	end

	% Close file
	fclose(fid);

end


% Convert device type to device type name
function name = getDeviceTypeName(deviceType)
	switch deviceType
		case 0,
			name = 'Cobra';
		case 4,
			name = 'Four-hole Cobra';
		case 5,
			name = 'Five-hole Cobra';
		case 13,
			name = 'Thirteen-hole ECA';
		case 1000,
			name = 'DP Module';
		case 1015,
			name = '15-channel DP Module';
		case 1016,
			name = '16-channel DP Module';
		case 1032,
			name = '32-channel DP Module';
		case 1064,
			name = '64-channel DP Module';
		case 1128,
			name = '128-channel DP Module';
		case 1256,
			name = '256-channel DP Module';
		case 8000,
			name = 'Force Balance';
		case 8010,
			name = 'JR3 Balance';
		case 8011,
			name = 'JR3 Balance (A/D)';
		case 8012,
			name = 'JR3 Balance (DSP)';
		case 8020,
			name = 'Aeroelastic Base Balance';
		case 10000,
			name = 'Analog Data Device';
		otherwise
			name = 'Unknown';
	end
end


% Convert device type to device type short name
function name = getDeviceTypeShortName(deviceType)
	switch deviceType
		case {0, 4, 5},
			name = 'Cobra';
		case 13,
			name = 'ECA';
		case {1000, 1015, 1016, 1032, 1064, 1128, 1256},
			name = 'DPM';
		case 8000,
			name = 'Force Balance';
		case {8010, 8011, 8012},
			name = 'JR3';
		case 8020,
			name = 'ABB';
		case 10000,
			name = 'ADD';
		otherwise
			name = 'Unknown';
	end
end


% Convert data type to data type name
function name = getDataTypeName(dataType)
	switch dataType
		case 0,
			name = 'Analogue';
		case 10,
			name = 'Voltage';
		case 20,
			name = 'Pressure';
		case 30,
			name = 'Temperature';
		case 100,
			name = 'Velocity';
		case 200,
			name = 'Force-Moment';
		otherwise
			name = 'Unknown';
	end
end


% Convert data order to data order name
function name = getDataOrderName(dataOrder)
	switch dataOrder
		case 0,
			name = 'Non-interleaved';
		case 1,
			name = 'Interleaved';
		otherwise
			name = 'Unknown';
	end
end


% Read a date and time
function dateTimeString = readDateTime(fid)
	dateString = readDate(fid);
	timeString = readTime(fid);

	dateTimeString = sprintf('%s  %s', dateString, timeString);
end


% Read a date
function dateString = readDate(fid)
	year = fread(fid, 1, 'int16');
	month = fread(fid, 1, 'int16');
	dayOfWeek = fread(fid, 1, 'int16');
	day = fread(fid, 1, 'int16');

	dateNumber = datenum(year, month, day);
	dateString = datestr(dateNumber, 'dd-mmm-yyyy');
end


% Read a time
function timeString = readTime(fid)
	hour = fread(fid, 1, 'int16');
	minute = fread(fid, 1, 'int16');
	second = fread(fid, 1, 'int16');
	milliSecond = fread(fid, 1, 'int16');

	timeNumber = datenum(0, 0, 0, hour, minute, second);
	timeString = datestr(timeNumber, 'HH:MM:SS');
	msString = sprintf('%0.3f', milliSecond/1000);
	msString = msString(2:end);
	timeString = strcat(timeString, msString);
end


% Determine number of samples per channel
function numSamples = getNumSamplesPerChannel(fid, numChannels)
	filePos = ftell(fid);
	fseek(fid,0,'eof');
	fileLength = ftell(fid);
	fseek(fid,filePos,'bof');
	numSamples = (fileLength - filePos) / (numChannels*4);
end

